const express = require('express');
const app = express();

app.set('view engine', 'ejs');
app.use(express.static('public'));
app.use(express.urlencoded({ extended: true }));

let items = [
    { id: '1', name: 'Milk', quantity: '2' },
    { id: '2', name: 'Bread', quantity: '1' },
    { id: '3', name: 'Eggs', quantity: '12' }
];
let nextId = 4;

function validateItem(name, quantity) {
  const errors = {};

  const cleanName = (name ?? "").trim();
  if (!cleanName) errors.name = "Item name is required.";
  else if (cleanName.length < 2) errors.name = "Item name must be at least 2 characters.";

  const qtyNum = Number(quantity);
  if (quantity === undefined || quantity === null || String(quantity).trim() === "") {
    errors.quantity = "Quantity is required.";
  } else if (!Number.isInteger(qtyNum) || qtyNum <= 0) {
    errors.quantity = "Quantity must be a whole number greater than 0.";
  }

  return { isValid: Object.keys(errors).length === 0, errors, cleanName, qtyNum };
}

function findItemById(id) {
  return items.find((it) => it.id === id);
}

app.get("/", (req, res) => {
  res.render("index", {
    title: "Shopping List - prabhjot kaur",
    items,
    errors: {},
    formData: { name: "", quantity: "" }
  });
});

app.post("/items", (req, res) => {
  const { name, quantity } = req.body;
  const v = validateItem(name, quantity);

  if (!v.isValid) {
    return res.status(400).render("index", {
      title: "Shopping List - prabhjot kaur",
      items,
      errors: v.errors,
      formData: { name: name ?? "", quantity: quantity ?? "" }
    });
  }

  items.push({ id: nextId++, name: v.cleanName, quantity: v.qtyNum });
  res.redirect("/");
});

app.get("/items/:id/edit", (req, res) => {
  const id = Number(req.params.id);
  const item = findItemById(id);

  if (!item) return res.status(404).send("Item not found");

  res.render("edit", {
    title: "Edit Item - prabhjot kaur",
    item,
    errors: {}
  });
});

app.post("/items/:id", (req, res) => {
  const id = Number(req.params.id);
  const item = findItemById(id);

  if (!item) return res.status(404).send("Item not found");

  const { name, quantity } = req.body;
  const v = validateItem(name, quantity);

  if (!v.isValid) {
    return res.status(400).render("edit", {
      title: "Edit Item - prabhjot kaur",
      item: { ...item, name: name ?? "", quantity: quantity ?? "" },
      errors: v.errors
    });
  }

  item.name = v.cleanName;
  item.quantity = v.qtyNum;

  res.redirect("/");
});

app.get("/items/:id/delete", (req, res) => {
  const id = Number(req.params.id);
  items = items.filter((it) => it.id !== id);
  res.redirect("/");
});


app.listen(3000, () => console.log('Server is running on http://localhost:3000'));
